defmodule Reseau.Etape do
  @moduledoc """
  Une étape abstraite décrivant un moment de l'expérience utilisateur dans le jeu en réseau.

  Chaque étape doit implémenter ce comportement.
  """

  @typedoc "Un joueur"
  @type joueur() :: Reseau.Joueur.t()

  @typedoc "Une partie"
  @type partie() :: Reseau.Partie.t()

  @doc """
  Callback appelée quand l'étape est chargée et prête à être affichée.

  Arguments :

  * `joueur` : le joueur actuel (`Reseau.Joueur`) ;
  * `partie` : la partie actuelle (`Reseau.Partie`).

  Elle doit retourner un tuple contenant le joueur et la partie (éventuellement modifiés).

  """
  @callback entrer(joueur(), partie()) :: {joueur(), partie()}

  @doc """
  Retourne le titre de l'étape.
  """
  @callback titre(joueur(), partie()) :: String.t()

  @doc """
  Retourne le texte de l'étape.
  """
  @callback texte(joueur(), partie()) :: String.t()

  @doc """
  Retourne le prompt de l'étape.
  """
  @callback prompt(joueur(), partie()) :: String.t()

  @doc """
  Gère les commandes entrées par l'utilisateur.

  La commande entrée est précisée sous la forme d'une chaîne de caractères. Le retour peut être de plusieurs types :

  - `:silence` : n'affiche rien et ne fait rien ;
  - `:prompt` : n'affiche rien sauf le prompt ;
  - `:rafraîchir` : n'affiche que l'écran de nouveau, sans message ;
  - `{:silence, message}` : affiche juste le message ;
  - `{:prompt, message}` : n'affiche que le message suivi du prompt ;
  - `{:rafraîchir, message}` : affiche le message suivi de tout l'écran ;
  - `{:rediriger, module, joueur, partie}` : change l'étape actuelle, affiche le nouvel écran ;
  - `{:rediriger, module, joueur, partie, message}` : change l'étape actuelle, affiche le message suivi du nouvel écran ;
  - `{:rediriger, module, joueur, partie, :silence}` : change l'étape actuelle, n'affiche rien ;
  - `{:rediriger, module, joueur, partie, :silence, message}` : change l'étape actuelle, affiche uniquement le message.

  """
  @callback gérer_commandes(joueur(), partie(), String.t()) ::
              :silence
              | :prompt
              | :rafrîchir
              | {:silence, String.t()}
              | {:prompt, String.t()}
              | {:rafraîchir, String.t()}
              | {:rediriger, module(), joueur(), partie()}
              | {:rediriger, module(), joueur(), partie(), String.t()}
              | {:rediriger, module(), joueur(), partie(), :silence}
              | {:rediriger, module(), joueur(), partie(), :silence, String.t()}
end
